'use strict'

// https://easings.net/#
const tweeners = {
    'hold': x => 0,
    'linear': x => x,
    'easeInSine': x => 1 - Math.cos((x * Math.PI) / 2),
    'easeOutSine': x => Math.sin((x * Math.PI) / 2),
    'easeInOutSine': x => -(Math.cos(Math.PI * x) - 1) / 2,
    'easeOutBounce': x => {
        const n1 = 7.5625
        const d1 = 2.75
        if (x < 1 / d1) {
            return n1 * x * x
        } else if (x < 2 / d1) {
            return n1 * (x -= 1.5 / d1) * x + 0.75
        } else if (x < 2.5 / d1) {
            return n1 * (x -= 2.25 / d1) * x + 0.9375
        } else {
            return n1 * (x -= 2.625 / d1) * x + 0.984375
        }
    }
}

function keyValidate_boolean (keyData) {
    return typeof keyData === 'boolean'
}
function keyLerp_boolean (a, b, alpha) {
    return a
}

function keyValidate_int (keyData) {
    return typeof keyData === 'number'
}
function keyLerp_int (a, b, alpha) {
    return Math.floor(a + (b - a) * alpha)
}
function keyLimits_int (paramConfig, values) {
    return [
        paramConfig.uiOptions?.min ?? Math.min(...values),
        paramConfig.uiOptions?.max ?? Math.max(...values)
    ]
}

function keyValidate_float (keyData) {
    return typeof keyData === 'number'
}
function keyLerp_float (a, b, alpha) {
    return a + (b - a) * alpha
}
function keyLimits_float (paramConfig, values) {
    return [
        paramConfig.uiOptions?.min ?? Math.min(...values),
        paramConfig.uiOptions?.max ?? Math.max(...values)
    ]
}

function keyValidate_float2 (keyData) {
    if (typeof keyData === 'object' &&
    keyData.length === 2 &&
        typeof keyData[0] === 'number' &&
        typeof keyData[1] === 'number') {
      return true
    } else {
      return false
    }
}
function keyLerp_float2 (a, b, alpha) {
    return [
        a[0] + (b[0] - a[0]) * alpha,
        a[1] + (b[1] - a[1]) * alpha,
    ]
}

function keyValidate_float3 (keyData) {
    if (typeof keyData === 'object' &&
    keyData.length === 3 &&
        typeof keyData[0] === 'number' &&
        typeof keyData[1] === 'number' &&
        typeof keyData[2] === 'number') {
      return true
    } else {
      return false
    }
}
function keyLerp_float3 (a, b, alpha) {
    return [
        a[0] + (b[0] - a[0]) * alpha,
        a[1] + (b[1] - a[1]) * alpha,
        a[2] + (b[2] - a[2]) * alpha,
    ]
}

function keyValidate_float4 (keyData) {
    if (typeof keyData === 'object' &&
    keyData.length === 4 &&
        typeof keyData[0] === 'number' &&
        typeof keyData[1] === 'number' &&
        typeof keyData[2] === 'number' &&
        typeof keyData[3] === 'number') {
      return true
    } else {
      return false
    }
}
function keyLerp_float4 (a, b, alpha) {
    return [
        a[0] + (b[0] - a[0]) * alpha,
        a[1] + (b[1] - a[1]) * alpha,
        a[2] + (b[2] - a[2]) * alpha,
        a[3] + (b[3] - a[3]) * alpha,
    ]
}

function keyValidate_string (keyData) {
    return typeof keyData === 'string'
}
function keyLerp_string (a, b, alpha) {
    return a
}

const allKeyFuncs = {
    boolean: [keyValidate_boolean, keyLerp_boolean, (x) => !!x],
    int: [keyValidate_int, keyLerp_int, (x) => Math.floor(x), keyLimits_int],
    float: [keyValidate_float, keyLerp_float, (x) => x, keyLimits_float],
    float2: [keyValidate_float2, keyLerp_float2, (x) => x],
    float3: [keyValidate_float3, keyLerp_float3, (x) => x],
    float4: [keyValidate_float4, keyLerp_float4, (x) => x],
    string: [keyValidate_string, keyLerp_string, (x) => x],
    angle: [keyValidate_float, keyLerp_float, (x) => degToRad(x)],
    angle3: [keyValidate_float3, keyLerp_float3, (x) => [degToRad(x[0]), degToRad(x[1]), degToRad(x[2])]],
    color: [keyValidate_float3, keyLerp_float3, (x) => x],
    model: [keyValidate_string, keyLerp_string, (x) => getModel(x)],
    texture: [keyValidate_string, keyLerp_string, (x) => getTexture(x)],
    colorTexture: [keyValidate_string, keyLerp_string, (x) => getColorTexture(x)],
    midiIn: [keyValidate_string, keyLerp_string, (x) => getColorTexture(x)],
    midiOut: [keyValidate_string, keyLerp_string, (x) => getColorTexture(x)],
    entityArray: [() => true, null, (x) => x],
}

function calculateDynamicConfig (time, configs, syncs, config, returnRawValue=false) {
    const results = {}
    config.forEach(configEntry => {
        const {
            paramName,
            type,
            defaultValue,
        } = { ...configEntry }
        results[paramName] = readAtTime(time, configs, syncs, paramName, type, defaultValue, returnRawValue)
    })
    return results
}

function calculateSamples(paramName, configs, config, t0, t1, tStep) {
    const paramConfig = config.find(x => x.paramName === paramName) 
    const {
        type,
        defaultValue,
    } = {...paramConfig}
    const keyFuncs = allKeyFuncs[type]
    const sampler = keyFuncs[1]

    const keyData = configs[paramName]
    if (keyData === undefined) {
        // No value set.
        let min, max
        if (keyFuncs[3]) {
            [min, max] = keyFuncs[3](paramConfig, [defaultValue])
        }
        return {
            type,
            max,
            min,
            startExtrapolated: true,
            endExtrapolated: true,
            times: [t0, t1],
            values: [defaultValue, defaultValue],
            handles: [],
        }
    } else if (keyData.type === 'value') {
        // Fixed value.
        let min, max
        if (keyFuncs[3]) {
            [min, max] = keyFuncs[3](paramConfig, [keyData.value])
        }
        return {
            type,
            max,
            min,
            startExtrapolated: true,
            endExtrapolated: true,
            times: [t0, t1],
            values: [keyData.value, keyData.value],
            handles: [],
        }
    } else {
        // Keyframes.
        let times = []
        let values = []
        let handles = []
        let min = 0
        let max = 0
        let startExtrapolated = false
        let endExtrapolated = false
        const numKeys = keyData.keys.length
        const [firstTime, firstValue] = keyData.keys[0]
        if (firstTime > t0) {
            times.push(t0)
            values.push(firstValue)
            startExtrapolated = true
        }
        for (let i = 0; i < numKeys - 1; ++i) {
            const [time, value, tweenerType] = keyData.keys[i]
            max = Math.max(max, value)
            min = Math.min(min, value)
            handles.push([time, value])
            switch (tweenerType) {
                case undefined:
                case 'linear':
                    times.push(time)
                    values.push(value)
                    break
                case 'hold':
                    {
                        times.push(time)
                        values.push(value)
                        const [timeNext] = keyData.keys[i + 1]
                        times.push(timeNext)
                        values.push(value)
                    }
                    break
                default:
                    {
                        const [timeNext, valueNext] = keyData.keys[i + 1]
                        const tRange = timeNext - time
                        let t = time
                        const tweener = tweeners[tweenerType]
                        while (t < timeNext) {
                            times.push(t)
                            const a = tweener((t - time) / tRange)
                            const v = sampler(value, valueNext, a)
                            values.push(v)
                            t += tStep
                        }
                    }
                    break
            }
        }
        const [finalTime, finalValue] = keyData.keys[numKeys - 1]
        times.push(finalTime)
        values.push(finalValue)
        if (finalTime < t1) {
            times.push(t1)
            values.push(finalValue)
            endExtrapolated = true
        }
        handles.push([finalTime, finalValue])

        if (keyFuncs[3]) {
            [min, max] = keyFuncs[3](paramConfig, values)
        }

        return {
            type,
            max,
            min,
            startExtrapolated,
            endExtrapolated,
            times,
            values,
            handles,
        }
    }
}

function readAtTime (time, configs, syncs, paramName, keyType, defaultValue, returnRawValue) {
    const keyFuncs = allKeyFuncs[keyType]

    let result = undefined
    const keyData = configs[paramName]
    if (keyData === undefined) {
        result = defaultValue
    } else if (keyData.type === 'value') {
        result = keyData.value
    } else if (keyData.type === 'keys') {
        var lastKey = keyData.keys[0]
        if (time <= lastKey[0]) {
            result = lastKey[1]
        } else {
            for (var i in keyData.keys) {
                const nextKey = keyData.keys[i]
                if (nextKey[0] > time) {
                    const alpha = (time - lastKey[0]) / (nextKey[0] - lastKey[0])
                    let tweener = lastKey[2] ? lastKey[2] : 'linear'
                    if (tweeners[tweener] === undefined) {
                        console.warn(`Unknown tweener type '${tweener}'`)
                        tweener = 'linear'
                    }
                    const tweenedAlpha = tweeners[tweener](alpha)
                    result = keyFuncs[1](lastKey[1], nextKey[1], tweenedAlpha)
                    break
                }
                lastKey = nextKey
            }
            if (result === undefined) {
                result = lastKey[1]
            }
        }
    } else {
        result = defaultValue
    }

    if (!returnRawValue) {
        result = keyFuncs[2](result)

        if (keyData?.expression) {
            try {
                const expressionResult = expressionParserCache.executeExpression(keyData.expression, {
                    self: result,
                    time: time,
                })
                if (expressionResult !== undefined) {
                    if (keyType === 'float') {
                        result = expressionResult
                    } else {
                        result = Math.round(expressionResult)
                    }
                }
            } catch {
            }
        }
    }

    return result
}

function calculateStaticConfig (instanceStaticConfig, entityStaticConfig) {
    const results = {}
    entityStaticConfig.forEach(configEntry => {
        const {
            paramName,
            defaultValue,
        } = { ...configEntry }
        const instanceStaticValue = instanceStaticConfig[paramName]
        if (instanceStaticValue !== undefined) {
            results[paramName] = instanceStaticValue
        } else {
            results[paramName] = defaultValue
        }
    })
    return results
}
